///////////////////////////////////////////////////////////////////////////////
// DCTO9PEMULATION.C - Emulateur Thomson TO9+ portable
// Author   : Daniel Coulom - danielcoulom@gmail.com
// Web site : http://dcto9p.free.fr
// Created  : December 2007
//
// This file is part of DCTO9P v11.
// 
// DCTO9P v11 is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// DCTO9P v11 is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with DCTO9P v11.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////


#include <string.h>
#include <time.h>
#include "../include/dcto9prom.h"
#include "dcto9pglobal.h"

// memory
char car[0x10000];   //espace cartouche 4x16K
char ram[0x80000];   //ram 512K 
char port[0x40];     //ports d'entree/sortie
char x7da[32];       //stockage de la palette de couleurs
// pointers
char *pagevideo;     //pointeur page video affichee
char *ramvideo;      //pointeur couleurs ou formes
char *ramuser;       //pointeur ram utilisateur fixe
char *rambank;       //pointeur banque ram utilisateur
char *romsys;        //pointeur rom systeme
char *rombank;       //pointeur banque rom ou cartouche
//banques
int nvideopage;      //n page video (00-01)
int nvideobank;      //n banque video (00-03)
int nrambank;        //n banque ram (00-1f)
int nrombank;        //n banque rom (00-07)
int nsystbank;       //n banque systeme (00-01)
int nctrlbank;       //n banque controleur (00-03)
//flags cartouche
int cartype;         //type de cartouche (0=simple 1=switch bank, 2=os-9)
int carflags;        //bits0,1,4=bank, 2=cart-enabled, 3=write-enabled
//keyboard, joysticks, mouse
int touche[TO9PKEY_MAX]; //etat touches to9p
int capslock;        //1=capslock, 0 sinon 
int joysposition;    //position des manches
int joysaction;      //position des boutons d'action
int xpen, ypen;      //lightpen coordinates
int penbutton;       //lightpen button state
//affichage
int videolinecycle;  //compteur ligne (0-63)
int videolinenumber; //numero de ligne video affiche (0-311)
int vblnumber;       //compteur du nombre de vbl avant affichage 
int vblnumbermax;    //nombre de vbl entre deux affichages 
int displayflag;     //indicateur pour l'affichage 
int bordercolor;     //couleur de la bordure de l'cran
//divers
int sound;           //niveau du haut-parleur
int mute;            // mute flag 
int timer6846;       //compteur du timer 6846
int latch6846;       //registre latch du timer 6846

//Accs mmoire
char MgetTO9P(unsigned short a);
void MputTO9P(unsigned short a, char c);
char (*Mgetc)(unsigned short);       //pointeur fonction courante
short Mgetw(unsigned short a) {return (Mgetc(a) << 8 | (Mgetc(++a) & 0xff));}
void (*Mputc)(unsigned short, char); //pointeur fonction courante
void Mputw(unsigned short a, short w) {Mputc(a, w >> 8); Mputc(++a, w);}
  
int to9key[0xa0] = { //Table de conversion scancode TO8 --> code ASCII
//code ASCII non shiftes
/*00*/0x91,0x5f,0x79,0x68,0x0b,0x09,0x1e,0x6e,
      0x92,0x28,0x74,0x67,0x3d,0x08,0x1c,0x62,
/*10*/0x93,0x27,0x72,0x66,0x16,0x31,0x1d,0x76,
      0x94,0x22,0x65,0x64,0x37,0x34,0x30,0x63,
/*20*/0x90,0x80,0x7a,0x73,0x38,0x32,0x2e,0x78,
      0x23,0x2a,0x61,0x71,0x5b,0x35,0x36,0x77,
/*30*/0x02,0x81,0x75,0x6a,0x20,0x39,0x0d,0x2c,
      0xb0,0x21,0x69,0x6b,0x24,0x0a,0x5d,0x3b,
/*40*/0xb7,0x82,0x6f,0x6c,0x2d,0x84,0x0d,0x3a,
      0xb3,0x83,0x70,0x6d,0x29,0x5e,0x33,0x3e, 
//code ASCII shiftes
/*00*/0x96,0x36,0x59,0x48,0x0b,0x09,0x0c,0x4e,
      0x97,0x35,0x54,0x47,0x2b,0x08,0x1c,0x42,
/*10*/0x98,0x34,0x52,0x46,0x16,0x31,0x7f,0x56,
      0x99,0x33,0x45,0x44,0x37,0x34,0x30,0x43,
/*20*/0x95,0x32,0x5a,0x53,0x38,0x32,0x2e,0x58,
      0x40,0x31,0x41,0x51,0x7b,0x35,0x36,0x57,
/*30*/0x03,0x37,0x55,0x4a,0x20,0x39,0x0d,0x3f,
      0xb0,0x38,0x49,0x4b,0x26,0x0a,0x7d,0x2e,
/*40*/0xb8,0x39,0x4f,0x4c,0x5c,0x25,0x0d,0x2f,
      0xb3,0x30,0x50,0x4d,0x86,0x85,0x33,0x3c}; 

//affichage
extern void (*Decodevideo)();     //pointeur fonction courante
extern void Decode320x16();       //standard TO
extern void Decode320x4();        //bitmap4
extern void Decode320x4special(); //bitmap4 special
extern void Decode160x16();       //bitmap16 
extern void Decode640x2();        //80 colonnes

// Emulation du clavier TO9 ///////////////////////////////////////////////////
void TO9key(int n)
{
 int i;
 if(touche[n]) //touche relachee
 {
  for(i = 0; i < 0x50; i++) if(touche[i] == 0) return;  //touche enfoncee
  port[0x08] = 0x00; //bit 0 de E7C8 = 0 (toutes les touches relachees)
  return;
 }
 //touche enfoncee
 if(n == 0x50) capslock = 1 - capslock; //capslock
 if(n > 0x4f) return;           //touches shift, ctrl et joysticks
 i = 0;
 if(!touche[0x51]) i = 0x80; //SHIFT gauche
 if(!touche[0x52]) i = 0x80; //SHIFT droit
 if(capslock) switch(n)
 {
  case 0x02: case 0x03: case 0x07: case 0x0a: case 0x0b: case 0x0f: case 0x12:
  case 0x13: case 0x17: case 0x1a: case 0x1b: case 0x1f: case 0x22: case 0x23:
  case 0x27: case 0x2a: case 0x2b: case 0x2f: case 0x32: case 0x33: case 0x3a:
  case 0x3b: case 0x42: case 0x43: case 0x4a: case 0x4b: i = 0x80; break;             
 }             
 if(i == 0x80) n += 0x50;
 i = to9key[n]; //code ASCII
 if(!touche[0x53]) //touche CNT
 {
  if(i == 23) i = 0;
  if((i > 0x3f) & (i < 0x60)) i -= 0x40;
  if((i > 0x60) & (i < 0x80)) i -= 0x60;   
 }       
 port[0x08] = 0x01; //bit 0 de E7C8 = 1 (touche enfoncee)
 port[0x1e] = 0x01; //touche en attente  
 port[0x1f] = i;    //code ASCII TO9-TO9+
}

// Selection de banques memoire //////////////////////////////////////////////
void TO8videoram()
{
 extern int nvideopage, nsystbank; 
 nvideopage = port[0x03] & 1;    
 ramvideo = ram - 0x4000 + (nvideopage << 13);
 nsystbank = (port[0x03] & 0x10) >> 4;
 romsys = to9prom + 0x2000 + (nsystbank << 13); 
}    

void TO8rambank()
{
 extern int nrambank;    
 extern void Initcontroller(); 
 //mode TO8 par e7e5   
 if(port[0x27] & 0x10)
 {
  nrambank = port[0x25] & 0x1f;              
  rambank = ram - 0xa000 + (nrambank << 14);
  return;
 }
 //mode compatibilite TO7/70 par e7c9   
 switch(port[0x09] & 0xf8)
 {
  case 0x08: nrambank = 0; break;
  case 0x10: nrambank = 1; break;
  case 0xe0: nrambank = 2; break;
  case 0xa0: nrambank = 3; break;  //banques 5 et 6
  case 0x60: nrambank = 4; break;  //inversees/TO770&TO9
  case 0x20: nrambank = 5; break;
  default: return;
 } 
 rambank = ram - 0x2000 + (nrambank << 14);
}

void TO8rombank()
{
 extern int nrombank;    
 //romsys = rom + 0x2000 + ((cnt[0x7c3] & 0x10) << 9); 
 //si le bit 0x20 de e7e6 est positionne a 1 l'espace ROM est recouvert
 //par la banque RAM definie par les 5 bits de poids faible de e7e6
 //subtilite : les deux segments de 8K de la banque sont inverss.
 if(port[0x26] & 0x20)
 {
  //e7c3 |= 4; 
  rombank = ram + ((port[0x26] & 0x1f) << 14);
  //rombank += (carflags & 3) << 14;
  return;
 }
 //sinon le bit 0x04 de e7c3 commute entre ROM interne et cartouche
 if((port[0x03] & 0x04) == 0)
 {
  nrombank = -1;
  rombank = car + ((carflags & 3) << 14);
 }
 else
 {
  nrombank = carflags & 3;   
  rombank = to9prom + (nrombank << 14);
 } 
}

void Switchmemo7(int a)
{
 carflags = (carflags & 0xfc) | (a & 3);
 rombank = car + ((carflags & 3) << 14);
}

void Videopage_bordercolor(char c)
{
 port[0x1d] = c;
 pagevideo = ram + ((c & 0xc0) << 8);
 bordercolor = c & 0x0f;
}

// Selection video ////////////////////////////////////////////////////////////
void TO8videomode(char c)
{
 port[0x1c] = c;
 switch(c)
 {
  case 0x21: Decodevideo = Decode320x4; break;
  case 0x2a: Decodevideo = Decode640x2; break;
  case 0x41: Decodevideo = Decode320x4special; break; 
  case 0x7b: Decodevideo = Decode160x16; break; 
  default:   Decodevideo = Decode320x16; break; 
 } 
}     

// Selection d'une couleur de palette /////////////////////////////////////////
void Palettecolor(char c)
{
 int i = port[0x1b];
 void Palette(int n, int r, int v, int b);
 x7da[i] = c;
 port[0x1b] = (port[0x1b] + 1) & 0x1f;
 if((i & 1))
 {
  char c1 = x7da[i & 0x1e];
  Palette(i >> 1, c1 & 0x0f, (c1 & 0xf0) >> 4, c & 0x0f);
 }
} 

// Signaux de synchronisation ligne et trame /////////////////////////////////
int Iniln()
{//11 microsecondes - 41 microsecondes - 12 microsecondes 
 if(videolinecycle < 11) return 0;   
 if(videolinecycle > 51) return 0;
 return 0x20;
}    

int Initn()
{//debut  12 microsecondes ligne 56, fin  51 microsecondes ligne 255
 if(videolinenumber < 56) return 0;
 if(videolinenumber > 255) return 0;
 if(videolinenumber == 56) if(videolinecycle < 12) return 0;
 if(videolinenumber == 255) if(videolinecycle > 50) return 0;
 return 0x80;
} 

// Joystick emulation ////////////////////////////////////////////////////////
void Joysemul(int i, int state)
{
 //PA0=0 nord   PA1=0 sud   PA2=0 ouest   PA3=0 est   PB6=0 action
 //PA4=1 nord   PA5=1 sud   PA6=1 ouest   PA7=1 est   PB7=1 action
 int n;
 n = 0;
 switch(i)
 {
  case 0: if(joysposition & 0x02) n = 0x01; break;        
  case 1: if(joysposition & 0x01) n = 0x02; break;        
  case 2: if(joysposition & 0x08) n = 0x04; break;        
  case 3: if(joysposition & 0x04) n = 0x08; break;        
  case 4: if(joysposition & 0x20) n = 0x10; break;        
  case 5: if(joysposition & 0x10) n = 0x20; break;        
  case 6: if(joysposition & 0x80) n = 0x40; break;        
  case 7: if(joysposition & 0x40) n = 0x80; break;        
  case 8: if(state) joysaction |= 0x40; else joysaction &= 0xbf; return;        
  case 9: if(state) joysaction |= 0x80; else joysaction &= 0x7f; return;        
 }
 if(n > 0) {if(state) joysposition |= n; else joysposition &= (~n);} 
}

// Joystick move /////////////////////////////////////////////////////////////
void Joysmove(int n, int x, int y)
{
 n <<= 2;
 Joysemul(n++, (y < 27768) ? 0 : 0x80);
 Joysemul(n++, (y > 37767) ? 0 : 0x80);
 Joysemul(n++, (x < 27768) ? 0 : 0x80);
 Joysemul(n++, (x > 37767) ? 0 : 0x80);
}

// Initialisation programme de l'ordinateur mul ////////////////////////////
void Initprog()
{
 int i;   
 extern int CC;
 extern short PC;
 short Mgetw(unsigned short a);
 for(i = 0; i < TO9PKEY_MAX; i++) touche[i] = 0x80; //touches relaches
 joysposition = 0xff;                      //manettes au centre
 joysaction = 0xc0;                        //boutons relachs
 carflags &= 0xec; 
 Mputc = MputTO9P;
 Mgetc = MgetTO9P;
 Decodevideo = Decode320x16;
 ramuser = ram - 0x2000;
 Videopage_bordercolor(port[0x1d]);
 TO8videoram();
 TO8rambank();
 TO8rombank();
 CC = 0x10;
 PC = Mgetw(0xfffe);
}

// Hardreset de l'ordinateur mul ///////////////////////////////////////////
void Hardreset()
{
 int i;
 time_t curtime;    
 struct tm *loctime;
 extern int pause6809;    
 pause6809 = 1;  
 for(i = 0; i < sizeof(ram); i++) ram[i] = -((i & 0x80) >> 7);
 for(i = 0; i < sizeof(port); i++) port[i] = 0; port[0x09] = 0x0f;
 for(i = 0; i < sizeof(car); i++) car[i] = 0;     
 //en rom : remplacer jj-mm-aa par la date courante
 curtime = time(NULL);
 loctime = localtime(&curtime);
 strftime(to9prom + 0xeb90, 9, "%d-%m-%y", loctime);
 to9prom[0xeb98] = 0x1f;
 //en rom : au reset initialiser la date courante
 //24E2 8E2B90  LDX  #$2B90
 //24E5 BD29C8  BSR  $29C8
 to9prom[0xe4e2] = 0x8e; to9prom[0xe4e3] = 0x2b; to9prom[0xe4e4] = 0x90;
 to9prom[0xe4e5] = 0xbd; to9prom[0xe4e6] = 0x29; to9prom[0xe4e7] = 0xc8;
 nvideobank = 0;      
 nrambank = 0;        
 nsystbank = 0;       
 nctrlbank = 0;       
 videolinecycle = 0;
 videolinenumber = 0;
 vblnumber = 0;
 Initprog();
 latch6846 = 65535;
 timer6846 = 65535;
 sound = 0;
 mute = 0;
 penbutton = 0;
 capslock = 1;
 pause6809 = 0;
}

// Timer control /////////////////////////////////////////////////////////////
void Timercontrol()
{
 if(port[0x05] & 0x01) timer6846 = latch6846 << 3;
}

// Traitement des entrees-sorties ////////////////////////////////////////////
void Entreesortie(int io)
{
 extern void Readoctetk7(), Writeoctetk7();
 extern void Readsector(), Writesector(), Formatdisk(), Imprime();
 extern void Readmousebutton(), Readpenxy(int i);
 switch(io)
 {
  case 0x14: Readsector(); break;      //lit secteur qd-fd
  case 0x15: Writesector(); break;     //ecrit secteur qd-fd
  case 0x18: Formatdisk(); break;      //formatage qd-fd
  case 0x42: Readoctetk7(); break;     //lit octet cassette
  case 0x45: Writeoctetk7(); break;    //ecrit octet cassette
  case 0x4b: Readpenxy(0); break;      //lit position crayon
  case 0x4e: Readpenxy(1); break;      //lit position souris
  case 0x51: Imprime(); break;         //imprime un caractere
  case 0x52: Readmousebutton(); break; //test clic souris
  default: break;                      //code op. invalide
 }
}

// Execution n cycles processeur 6809 ////////////////////////////////////////
int Run(int ncyclesmax)
{
 int ncycles, opcycles;
 extern int Run6809();
 extern void Irq();
 extern void Displaysegment();
 extern void Nextline();
 extern void Displayscreen();
 ncycles = 0;
 while(ncycles < ncyclesmax)
 {
  //execution d'une instruction
  opcycles = Run6809();
  if(opcycles < 0) {Entreesortie(-opcycles); opcycles = 64;}
  ncycles += opcycles;
  videolinecycle += opcycles;
  if(displayflag) Displaysegment();                      
  if(videolinecycle >= 64)
  {
   videolinecycle -= 64;
   if(displayflag) Nextline();
   if(++videolinenumber > 311)
   //valeurs de videolinenumber :
   //000-047 hors ecran, 048-055 bord haut
   //056-255 zone affichable
   //256-263 bord bas, 264-311 hors ecran
   {
    videolinenumber -= 312;
    if(vblnumber == 0) Displayscreen();
    if(++vblnumber >= vblnumbermax) vblnumber = 0; 
   }
   displayflag = 0;
   if(vblnumber == 0) 
     if(videolinenumber > 47)
       if(videolinenumber < 264)
         displayflag = 1;                      
  }     
  //timer 6846 
  if(timer6846 < ((latch6846 << 3) - 256))
  {port[0] &= 0xfe;} //reset IRQ flag (ce n'est pas conforme a la specification)
  if((port[0x05] & 0x01) == 0) //timer enabled
  {timer6846 -= (port[0x05] & 0x04) ? opcycles : opcycles << 3;} //countdown
  if(timer6846 <= 5) //counter time out
  {port[0] |= 1; Irq(); timer6846 = latch6846 << 3;} //IRQ, reset counter
 }
 return(ncycles - ncyclesmax); //retour du nombre de cycles en trop (extracycles)
}  

// Ecriture memoire to9p /////////////////////////////////////////////////////
void MputTO9P(unsigned short a, char c)
{
 switch(a >> 12)
 {
  //subtilit : 
  //quand la rom est recouverte par la ram, les 2 segments de 8 Ko sont inverss
  case 0x0: case 0x1:
   if(!(port[0x26] & 0x20)) {carflags = (carflags & 0xfc) | (a & 3); TO8rombank();}     
   if((port[0x26] & 0x60) != 0x60) return;
   if(port[0x26] & 0x20) rombank[a + 0x2000] = c; else rombank[a] = c; return; 
  case 0x2: case 0x3: if((port[0x26] & 0x60) != 0x60) return;
   if(port[0x26] & 0x20) rombank[a - 0x2000] = c; else rombank[a] = c; return; 
  case 0x4: case 0x5: ramvideo[a] = c; return;
  case 0x6: case 0x7: case 0x8: case 0x9: ramuser[a] = c; return;
  case 0xa: case 0xb: case 0xc: case 0xd: rambank[a] = c; return;
  case 0xe: switch(a)
  {
   case 0xe7c1: port[0x01] = c; mute = c & 0x08; return; 
   //case 0xe7c1: port[0x01] = c & 0x08; sound = (c & 0x08) << 2; return;
   case 0xe7c3: port[0x03] = (c & 0x3d); TO8videoram(); TO8rombank(); return;
   case 0xe7c5: port[0x05] = c; Timercontrol(); return; //controle timer
   case 0xe7c6: latch6846 = (latch6846 & 0xff) | ((c & 0xff) << 8); return;
   case 0xe7c7: latch6846 = (latch6846 & 0xff00) | (c & 0xff); return;
   case 0xe7c9: port[0x09] = c; TO8rambank(); return;
   case 0xe7cc: port[0x0c] = c; return;
   case 0xe7cd: if(port[0x0f] & 4) sound = c & 0x3f; else port[0x0d] = c; return;
   case 0xe7ce: port[0x0e] = c; return; //registre controle position joysticks
   case 0xe7cf: port[0x0f] = c; return; //registre controle action - musique
   case 0xe7d8: return;
   case 0xe7da: Palettecolor(c); return;
   case 0xe7db: port[0x1b] = c; return; 
   case 0xe7dc: TO8videomode(c); return;
   case 0xe7dd: Videopage_bordercolor(c); return;
   case 0xe7e4: port[0x24] = c; return;
   case 0xe7e5: port[0x25] = c; TO8rambank(); return;
   case 0xe7e6: port[0x26] = c; TO8rombank(); return;
   case 0xe7e7: port[0x27] = c; TO8rambank(); return;
   default: return; 
  } 
  default: return;
 }
 return; 
}  

// Lecture memoire to9p //////////////////////////////////////////////////////
char MgetTO9P(unsigned short a)
{
 extern int penbutton;
 extern int joysposition, joysaction;
 extern short PC;
 switch(a >> 12)
 {
  //subtilit : quand la rom est recouverte par la ram, les 2 segments de 8 Ko sont inverss
  case 0x0: case 0x1: return (port[0x26] & 0x20) ? rombank[a + 0x2000] : rombank[a];
  case 0x2: case 0x3: return (port[0x26] & 0x20) ? rombank[a - 0x2000] : rombank[a];
  case 0x4: case 0x5: return ramvideo[a];
  case 0x6: case 0x7: case 0x8: case 0x9: return ramuser[a];
  case 0xa: case 0xb: case 0xc: case 0xd: return rambank[a];
  case 0xe: switch(a)
  {
   //e7c0= 6846 composite status register
   //csr0= timer interrupt flag
   //csr1= cp1 interrupt flag (clavier)
   //csr2= cp2 interrupt flag
   //csr3-csr6 unused and set to zeroes
   //csr7= composite interrupt flag (si au moins un interrupt flag  1) 
   case 0xe7c0: return((port[0]) ? (port[0] | 0x80) : 0);
   case 0xe7c3: if((PC & 0xffff) == 0xf0e8) port[0x00] &= 0xfd;  //reset interruption clavier
        return(port[0x03] | 0x80 | (penbutton << 1));
   case 0xe7c6: return (timer6846 >> 11 & 0xff);
   case 0xe7c7: return (timer6846 >> 3 & 0xff);
   case 0xe7ca: return (videolinenumber < 200) ? 0 : 2; //non, registre de controle PIA
   case 0xe7cc: return((port[0x0e] & 4) ? joysposition : port[0x0c]);
   case 0xe7cd: return((port[0x0f] & 4) ? joysaction | sound : port[0x0d]);
   case 0xe7ce: return 0x04;
   case 0xe7da: return x7da[port[0x1b]++ & 0x1f];
   case 0xe7df: port[0x1e] = 0; return(port[0x1f]);
   case 0xe7e4: return port[0x1d] & 0xf0; 
   case 0xe7e5: return port[0x25] & 0x1f;
   case 0xe7e6: return port[0x26] & 0x7f;
   case 0xe7e7: return (port[0x24] & 0x01) | Initn() | Iniln();
   default: if(a < 0xe7c0) return(romsys[a]);
            if(a < 0xe800) return(port[a & 0x3f]);
            return(romsys[a]);
  }
  default: return romsys[a];
 }
}
